<?php

namespace App\Http\Controllers\Admin;

use App\Models\Medias;
use App\Models\Setting;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Classes\S3MediaStorage;
use App\Classes\LocalMediaStorage;
use App\Classes\WasabiMediaStorage;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use App\Classes\BackblazeMediaStorage;
use Illuminate\Support\Facades\Storage;

class MediaController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Show the application dashboard.
     *
     * @return \Illuminate\Contracts\Support\Renderable
     */

    // All user media
    public function media()
    {
        // Get media images
        $media = Medias::where('user_id', Auth::user()->id)->where('status', '1')->orderBy('id', 'desc')->paginate(8);
        $settings = Setting::where('status', 1)->first();

        // View media
        return view('admin.pages.media.index', compact('media', 'settings'));
    }

    // Add media
    public function addMedia()
    {
        // Queries
        $settings = Setting::where('status', 1)->first();

        // Add media
        return view('admin.pages.media.add', compact('settings'));
    }

    // Upload media
    public function uploadMedia(Request $request)
    {
        if (env('AWS_ENABLE') == 'on') {
            // Upload local
            $image = $request->file('file');

            // Upload AWS S3
            $path = Storage::disk('s3')->put('files/' . Auth::user()->id . '/media', $image);
            $request->merge([
                'size' => $image->getSize(),
                'path' => $path
            ]);

            $mediaImage = $request->only('path', 'title', 'size');

            $media = new Medias();
            $media->media_id = random_int(000000000000001, 99999999999999);
            $media->user_id = Auth::user()->id;
            $media->media_name = $image->getClientOriginalName();
            $media->media_url = Storage::disk('s3')->url($path);
            $media->size = $image->getSize();
            $media->save();
        } else if (env('WAS_ENABLE') == 'on') {
            // Upload local
            $image = $request->file('file');

            // Upload AWS wasabi
            $path = Storage::disk('wasabi')->put('files/' . Auth::user()->id . '/media', $image);
            $request->merge([
                'size' => $image->getSize(),
                'path' => $path
            ]);

            $mediaImage = $request->only('path', 'title', 'size');

            $media = new Medias();
            $media->media_id = random_int(000000000000001, 99999999999999);
            $media->user_id = Auth::user()->id;
            $media->media_name = $image->getClientOriginalName();
            $media->media_url = Storage::disk('wasabi')->url($path);
            $media->size = $image->getSize();
            $media->save();
        } else if (env('BACK_ENABLE') == 'on') {
            // Upload local
            $image = $request->file('file');

            // Upload AWS backblaze
            $path = Storage::disk('backblaze')->put('files/' . Auth::user()->id . '/media', $image);
            $request->merge([
                'size' => $image->getSize(),
                'path' => $path
            ]);

            $mediaImage = $request->only('path', 'title', 'size');

            $media = new Medias();
            $media->media_id = random_int(000000000000001, 99999999999999);
            $media->user_id = Auth::user()->id;
            $media->media_name = $image->getClientOriginalName();
            $media->media_url = Storage::disk('backblaze')->url($path);
            $media->size = $image->getSize();
            $media->save();
        } else {
            // Upload local
            $image = $request->file('file');

            // Image name
            $imageName = random_int(000000000000001, 99999999999999) . '.' . $image->extension();
            $movedImage = $image->move(public_path('/storage/media/files/' . Auth::user()->id), $imageName);

            $media = new Medias();
            $media->media_id = random_int(000000000000001, 99999999999999);
            $media->user_id = Auth::user()->id;
            $media->media_name = $image->getClientOriginalName();
            $media->media_url = "/storage/media/files/" . Auth::user()->id . '/' . $imageName;
            $media->size = $movedImage->getSize();
            $media->save();
        }

        return response()->json(['status' => 'success', 'message' => trans('Image has been successfully uploaded.')]);
    }

    // Optimize image
    public function optimizeImage($galleryId)
    {
        // Get gallery details
        $mediaDetails = Medias::where('id', $galleryId)->where('user_id', Auth::user()->id)->first();

        // Check s3 available
        if (env('AWS_ENABLE') == 'on' && substr($mediaDetails->media_url, 0, 5) != "https") {
            $storage = new S3MediaStorage;
            $storage->s3MediaFunction($mediaDetails);
        } else if (env('WAS_ENABLE') == 'on' && substr($mediaDetails->media_url, 0, 5) != "https") {
            $storage = new WasabiMediaStorage;
            $storage->wasabiMediaFunction($mediaDetails);
        } else if (env('BACK_ENABLE') == 'on' && substr($mediaDetails->media_url, 0, 5) != "https") {
            $storage = new BackblazeMediaStorage;
            $storage->backBlazeMediaFunction($mediaDetails);
        } else {
            $storage = new LocalMediaStorage;
            $storage->localMediaFunction($mediaDetails);
        }

        return redirect()->route('admin.media')->with('success', trans('Image has been optimized successfully. Please check your optimized image in your Optimizer page.'));
    }

    // Delete media
    public function deleteMedia($mid)
    {
        $media_data = Medias::where('user_id', Auth::user()->id)->where('id', $mid)->first();
        if ($media_data != null) {
            // Get media details
            $mediaDetails = Medias::where('user_id', Auth::user()->id)->where('id', $mid)->first();

            // Delete (local) media image
            $optimized_image = str_replace('/storage/media/files/' . Auth::user()->id . '/', '', $mediaDetails->media_url);

            // Check s3
            if (substr($optimized_image, 0, 5) == "https") {
                try {
                    // Delete s3 image
                    Storage::disk('s3')->delete(str_replace(Storage::disk('s3')->url('.'), '', $mediaDetails->media_url));
                } catch (\Throwable $th) {
                    //throw $th;
                }

                // Delete wasabi image
                Storage::disk('wasabi')->delete(str_replace(Storage::disk('wasabi')->url('.'), '', $mediaDetails->media_url));
            } else {
                unlink(public_path('storage/media/files/' . Auth::user()->id . '/' . $optimized_image));
            }

            // Delete media
            Medias::where('user_id', Auth::user()->id)->where('id', $mid)->delete();

            // Page redirect
            return redirect()->route('admin.media')->with('success', trans('Media file removed!'));
        } else {
            // Page redirect
            return redirect()->route('admin.media')->with('failed', trans('Media not found!'));
        }
    }

    // Bulk Delete
    public function bulkDeleteMedia(Request $request)
    {
        // Update status
        Medias::whereIn('id', explode(',', $request->query('id')))->where('user_id', Auth::user()->id)->update(['status' => 0]);

        return redirect()->route('admin.media')->with('success', trans('Media Deleted Successfully!'));
    }

    // Bulk Download
    public function bulkDownloadMedia(Request $request)
    {
        // Queries
        $optimize_details = Medias::whereIn('id', explode(',', $request->query('id')))->where('user_id', Auth::user()->id)->get();

        $downloads = [];
        $folderName = str_replace(" ", "-", config('app.name'));
        $zipName = Str::lower($folderName) . '-' . uniqid() . '.zip'; // Name of our archive to download

        // Initializing PHP class
        $zip = new \ZipArchive();
        $zip->open($zipName, \ZipArchive::CREATE | \ZipArchive::OVERWRITE);

        if (isset($optimize_details)) {
            for ($i=0; $i < count($optimize_details); $i++) { 
                if (substr($optimize_details[$i]->media_url, 0, 5) != "https") {
                    // Download optimized image
                    $file_name = public_path($optimize_details[$i]->media_url);
    
                    $zip->addFile($file_name, $optimize_details[$i]->media_name);
                } else {
                    // Download optimized image
                    $path = public_path($optimize_details[$i]->media_url);
                    Storage::disk('local')->put($optimize_details[$i]->media_name, file_get_contents($path));
    
                    // Path
                    $downloads[$i] = Storage::path($optimize_details[$i]->media_name);

                    // Adding file: second parameter is what will the path inside of the archive
                    // So it will create another folder called "storage/" inside ZIP, and put the file there.
                    $zip->addFile($downloads[$i], $optimize_details[$i]->media_name);

                }
            }
        } else {
            return redirect()->route('admin.media')->with('failed', trans('Image Not Found!'));
        }

        $zip->close();

        // We return the file immediately after download
        return response()->download($zipName);
    }
}
 